package vital_ecg;

import java.util.LinkedList;
import java.util.Queue;
import java.util.Random;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import data.ECGConst;
import javafx.application.Platform;

import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.concurrent.Task;
import javafx.scene.chart.AreaChart;
import javafx.scene.chart.NumberAxis;
import javafx.scene.chart.XYChart;

/**
 * Graf pro vykreslovani CO2.
 *
 * @author Patrik Patera
 * @version 2.00
 */
public class CO2Chart extends AreaChart<Number, Number> {

    /**
     * Seznam vykreslovacich dat.
     */
    private final ObservableList<Data<Number, Number>> dataPlot;

    /**
     * Prirazeni vykreslovacich dat do grafu.
     */
    private final XYChart.Series<Number, Number> series;

    /**
     * Fronta pridavanych casu pro jejich plynule vykreslovani.
     */
    private final Queue<Float> timeQueue;

    /**
     * Ziskani X-ove osy pro plynuly pohyb grafu.
     */
    private final NumberAxis xAxis;

    /**
     * Manager na spusteni vlaken.
     */
    private final ExecutorService ex;

    /**
     * Vykreslovani grafu.
     */
    private Task animation;

    /**
     * Simulace nadechu a vydechu.
     */
    private Task breath;

    /**
     * Prirazena hodnota CO2 z ovladace.
     */
    private float CO2Value;

    /**
     * Hodnota CO2.
     */
    private float tmpCO2Value;

    /**
     * Cas pri vykreslovani grafu.
     */
    private float timeX;

    /**
     * Cekaci doba pri vykreslovni.
     */
    private double WAIT;

    /**
     * Priznak, jestli se ma vykreslovat vydech u CO2.
     */
    private boolean plotting;

    /**
     * Priznak, zda-li ma probihat animace vykeslovani grafu.
     */
    private boolean hasAnimation;

    /**
     * Konstruktor, ktery na zaklade prijatych souradnic vytvori graf pro
     * zobrazovani CO2.
     *
     * @param axis  X-ova souradna osa.
     * @param axis1 Y-ova souradna osa.
     */
    public CO2Chart(NumberAxis axis, NumberAxis axis1) {
        super(axis, axis1);

        this.dataPlot = FXCollections.observableArrayList();
        this.series = new AreaChart.Series<>(dataPlot);
        this.xAxis = axis;
        this.ex = Executors.newWorkStealingPool();
        this.timeQueue = new LinkedList<>();

        this.getData().add(series);
        this.setAnimated(false);
        this.setCreateSymbols(false);
        this.hasAnimation = true;

        breathing();
        animation();
        initValues();
    }

    /**
     * Nastaveni pocatecnich hodnot pro vykreslovani grafu.
     */
    private void initValues() {
        this.CO2Value = 0;
        this.timeX = 0;
        WAIT = ECGConst.DURATION_SECONDS / (ECGConst.DURATION_SECONDS / ECGConst.CO2_VZOREK)
                * ECGConst.SEC_TO_NANOSEC;
    }

    /**
     * Nastaveni vlakna pro primitivni simulaci dychani.
     * nadechu/vydechu.
     */
    private void breathing() {
        final Random rand = new Random();

        breath = new Task() {
            int time = 1;

            @Override
            protected Void call() throws Exception {
                while (true) {
                    if (isCancelled()) {
                        break;
                    }

                    plotting = !plotting;

                    tmpCO2Value = plotting ? CO2Value : 0;

                    time = rand.nextInt((2 - 1) + 1) + 1;

                    TimeUnit.SECONDS.sleep(time);
                }

                return null;
            }
        };

        ex.submit(breath);
    }

    /**
     * Nastaveni vlakna pro animaci vykreslovani grafu.
     */
    private void animation() {
        animation = new Task() {
            @Override
            protected Void call() throws Exception {
                while (true) {
                    if (isCancelled()) {
                        breath.cancel();
                        break;
                    }

                    if (!hasAnimation) continue;

                    synchronized (timeQueue) {
                        timeQueue.add(-timeX);
                    }

                    Platform.runLater(() -> {
                        synchronized (timeQueue) {
                            dataPlot.add(0, new XYChart.Data<>(timeQueue.remove(), tmpCO2Value));
                        }

                        if (dataPlot.size() > ECGConst.MAX_C02_POINTS) {
                            dataPlot.remove(ECGConst.MAX_C02_POINTS, dataPlot.size() - 1);
                        }
                    });

                    xAxis.setLowerBound(-timeX);
                    xAxis.setUpperBound(-timeX + ECGConst.DURATION_SECONDS);
                    TimeUnit.NANOSECONDS.sleep((long) WAIT);

                    timeX += ECGConst.CO2_VZOREK;

                    if (timeX >= 10e6) {
                        timeX = ECGConst.DURATION_SECONDS;
                    }
                }

                return null;
            }
        };

        ex.submit(animation);
    }

    /**
     * Nastaveni hodnoty CO2 pro vykreslovani grafu.
     *
     * @param value nastavovana hodnota CO2.
     */
    public void setCO2(int value) {
        if (value > 0) {
            this.CO2Value = value;
        }
    }

    /**
     * Uzavreni vsechn bezicich vlaken.
     */
    public void die() {
        animation.cancel();
        breath.cancel();
        ex.shutdownNow();
    }

    /**
     * Nastaveni, zda-li ma probihat animace vykreslovani krivek.
     *
     * @param flag priznak animace.
     */
    public void setHasAnimation(boolean flag) {
        hasAnimation = flag;
    }

}
