package vital_monitor;

import javafx.animation.Animation;
import javafx.animation.KeyFrame;
import javafx.animation.Timeline;
import javafx.application.Platform;
import javafx.scene.Node;
import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.ProgressBar;
import javafx.scene.layout.GridPane;
import javafx.util.Duration;
import language.KeysForLanguage;
import language.LoadFXMLWithResourceBundle;
import vital_monitor.settings.ValuesAboutStateOfApplication;

/**
 * Created by BobrZlosyn on 9. 4. 2016.
 */
public class DefibrilatorHandle {

    private static final String buttonEnergyStyle = "-fx-background-color:white; -fx-text-fill:black;";
    private static final String buttonImpedanceStyle = "-fx-background-color:white; -fx-text-fill:black;";

    private Timeline reloading, countingTimeLast, countingTimeTotal, analyseAnimation;
    private GridPane defibrilator, buttonsImpedance, buttonsEnergy;
    private Label typeOfDefibrilator, power, numberOfShocksLabel, timeLastLabel, timeTotalLabel, infoLabel;
    private ProgressBar reload, analyse;
    private Button shock, closeDefibrilator, activeEnergy;

    private int numberOfShocks, timeLast, timeTotal, defaultVolumeSound, impedance;
    private double powerValue;
    private boolean startOfDef, isAED, analysaResult, isAnalysaComplete, terapyElectrods, electrods, activeSounds;
    private PlaySounds playSounds;

    public DefibrilatorHandle(GridPane defibrilator, Label typeOfDefibrilator, Label power, Label numberOfShocksLabel,
                              Label timeLastLabel, Label timeTotalLabel, ProgressBar reload, Button shock,
                              Label infoLabel, ProgressBar analyse, Button closeDefibrilator, GridPane buttonsImpedance,
                              GridPane buttonsEnergy) {

        this.defaultVolumeSound = 5;
        this.defibrilator = defibrilator;
        this.typeOfDefibrilator = typeOfDefibrilator;
        this.power = power;
        this.numberOfShocksLabel = numberOfShocksLabel;
        this.timeLastLabel = timeLastLabel;
        this.timeTotalLabel = timeTotalLabel;
        this.reload = reload;
        this.shock = shock;
        this.startOfDef = false;
        this.infoLabel = infoLabel;
        this.analyse = analyse;
        this.playSounds = new PlaySounds(defaultVolumeSound);
        this.closeDefibrilator = closeDefibrilator;
        this.terapyElectrods = false;
        this.electrods = false;
        this.activeSounds = true;
        this.buttonsImpedance = buttonsImpedance;
        this.buttonsEnergy = buttonsEnergy;
    }

    public void setActiveSounds(boolean isActive) {
        activeSounds = isActive;
        playSounds.setVolume(!isActive ? 0 : defaultVolumeSound);
    }

    /**
     * Set the power of discharge in defibrillator.
     *
     * @param active actually active button with the power.
     */
    public void setActivePowerButton(Button active) {
        for (final Node node : buttonsEnergy.getChildren()) {
            if (active.equals(node)) {
                node.setStyle(buttonEnergyStyle);
                activeEnergy = (Button) node;
            } else {
                node.setStyle("");
            }
        }

        powerValue = Double.valueOf(activeEnergy.getText().split(" ")[0]);
        power.setText(activeEnergy.getText());
    }

    public boolean isActiveSound() {
        return activeSounds;
    }

    /**
     * zobrazi defibrilator v pozadovanem modu
     *
     * @param manualSetting
     * @param AEDfunction
     */
    public void defibStart(boolean AED, GridPane manualSetting, GridPane AEDfunction) {
        String mode;
        if (AED) {
            mode = LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_MODE_AED_LONG);
            changeElectrods();
        } else {
            mode = LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_MODE_MANUAL_LONG);
            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_MENU_ENERGY));
        }

        setModeDefibrilator(AED, mode, manualSetting, AEDfunction);

        ValuesAboutStateOfApplication.isDefibrilatorManualOn = !AED;
        ValuesAboutStateOfApplication.isDefibrilatorAEDOn = AED;

        showDefibrilator();
    }

    /**
     * spousti animaci nacitani energie pro vyboj
     *
     * @param isManual true pokud je manualni
     */
    public void reloadDefibrilator(boolean isManual) {
        if (reloading == null && startOfDef) {

            disablePowerButtons(true);
            double time = powerValue / 1000 * 2;

            if (isManual) {
                infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_CHARGE));
            }

            changeCloseButton(true);
            reloading = new Timeline(new KeyFrame(Duration.seconds(time), e -> reloadDefRun()));
            reloading.setCycleCount(Animation.INDEFINITE);
            reloading.play();
        }
    }

    /**
     * provede vyboj a zacne pocitat cas od probehnuti vyboje
     */
    public void startShock() {
        Platform.runLater(() -> {
            reload.setProgress(0);
            numberOfShocksLabel.setText(++numberOfShocks + "");
            timeLast = 0;
            timeLastLabel.setText("00:00");
            shock.setDisable(true);
            disablePowerButtons(false);
            disableImpedanceButtons(false);
            changeCloseButton(false);

            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_SHOCK_GIVEN));
            playSounds.stopShockReadyAnimation();
            playSounds.playSound(playSounds.SHOCK_GIVEN);
        });

        if (countingTimeLast == null) {
            countingTimeLast = new Timeline(new KeyFrame(Duration.seconds(1), e -> setTimeLast()));
            countingTimeLast.setCycleCount(Animation.INDEFINITE);
            countingTimeLast.play();
        }
    }

    /**
     * nastavi elektrody pri prichodu novych dat z ovladace
     *
     * @param electrods
     * @param terapyElectrods
     */
    public void setElectrods(boolean electrods, boolean terapyElectrods) {
        if (electrods != this.electrods || terapyElectrods != this.terapyElectrods) {
            this.electrods = electrods;
            this.terapyElectrods = terapyElectrods;

            if (defibrilator.isVisible()) {
                changeElectrods();
            }
        }
    }

    /**
     * pocita celkovy cas v timeline countingTimeTotal
     */
    private void setTimeTotal() {
        timeTotalLabel.setText(String.format("%02d:%02d", timeTotal / 60, timeTotal % 60));
        timeTotal++;
    }

    /**
     * pocita celkovy cas v timeline countingTimeLast
     */
    private void setTimeLast() {
        int minutes = timeLast / 60;
        int seconds = timeLast % 60;

        if (minutes == 0 && seconds == 1 && !isAED) {
            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_MENU_ENERGY));
        }

        if (minutes == 0 && seconds == 4 && isAED && analyseAnimation == null) {
            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_KPR));
            playSounds.playSound(PlaySounds.DO_KPR);
        }

        if (minutes == 2 && seconds == 0 && isAED) {
            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_START_ANALYSE));
            playSounds.playSound(PlaySounds.START_ANALYSE_EKG);
        }

        timeLastLabel.setText(String.format("%02d:%02d", timeLast / 60, timeLast % 60));
        timeLast++;
    }

    /**
     * zobrazi defibrilator na monitoru
     */
    public void showDefibrilator() {
        if (countingTimeTotal == null) {
            defibrilator.setVisible(true);
            startOfDef = true;

            Platform.runLater(() -> {
                powerValue = 200;
                power.setText((int) powerValue + " J");
                numberOfShocks = 0;
                numberOfShocksLabel.setText(numberOfShocks + "");
                timeLast = 0;
                timeLastLabel.setText("00:00");
                timeTotal = 1;
                timeTotalLabel.setText("00:01");
            });

            countingTimeTotal = new Timeline(new KeyFrame(Duration.seconds(1), e -> setTimeTotal()));
            countingTimeTotal.setCycleCount(Animation.INDEFINITE);
            countingTimeTotal.play();
        }
    }

    /**
     * skryje defibrilator ze sceny
     */
    public void hideDefibrilator() {
        ValuesAboutStateOfApplication.isDefibrilatorAEDOn = false;
        ValuesAboutStateOfApplication.isDefibrilatorManualOn = false;

        defibrilator.setVisible(false);
        startOfDef = false;

        countingTimeLast = stopAnimation(countingTimeLast);
        countingTimeTotal = stopAnimation(countingTimeTotal);
    }

    /**
     * zmena stavu defibrilatoru podle kliknuti na checkbox pro elektrody
     */
    public void changeElectrods() {
        if (!setElectrodsAction()) return;

        if (!changeTerapyElectrods()) return;

        if (isAED) {
            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_START_ANALYSE));
            playSounds.playSound(PlaySounds.START_ANALYSE_EKG);
        } else {
            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_MENU_ENERGY));
        }
    }

    /**
     * zjisti zda je nastavena elektroda pro terapii
     *
     * @return true pokud byla elektroda připojena
     */
    private boolean changeTerapyElectrods() {
        if (!terapyElectrods) {
            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_FREE_THERAPY));
            playSounds.playSound(PlaySounds.CONNECT_ELECTRODS_TERAPY);
            return false;
        }

        return true;
    }

    /**
     * spousti analyzu v defibrilatoru
     *
     * @param result true pokud ma byt analyza uspesna
     */
    public void runAnalyza(boolean result) {
        if (!electrods || !terapyElectrods || !isAED || analyseAnimation != null) {
            return;
        }
        infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_DONT_TOUCH));
        playSounds.playSound(PlaySounds.DONT_TOUCH_PACIENT);

        isAnalysaComplete = false;
        this.analysaResult = result;

        changeCloseButton(true);
        analyseAnimation = new Timeline(new KeyFrame(Duration.seconds(0.5), e -> analyseAnimation()));
        analyseAnimation.setCycleCount(Animation.INDEFINITE);
        analyseAnimation.play();
    }

    /**
     * nastaveni impedance s oznacenim aktivni impedance
     *
     * @param impedance 2- vysoka, 1 - stredni, 0 - nizka
     */
    public void setImpedanceFromRemoter(int impedance) {
        for (int i = 0; i < buttonsImpedance.getChildren().size(); i++) {
            if (impedance == i) {
                buttonsImpedance.getChildren().get(i).setStyle(buttonImpedanceStyle);
            } else {
                buttonsImpedance.getChildren().get(i).setStyle("");
            }
        }

        this.impedance = impedance;
    }

    /**
     * Energy buttons set disable for clicking on them.
     *
     * @param disable true - disable energy buttons
     */
    private void disablePowerButtons(boolean disable) {
        for (final Node node : buttonsEnergy.getChildren()) {
            node.setDisable(disable);
        }
    }

    /**
     * Impedance buttons set disable for clicking on them.
     *
     * @param disable true - disable impedance buttons
     */
    private void disableImpedanceButtons(boolean disable) {
        for (final Node node : buttonsImpedance.getChildren()) {
            node.setDisable(disable);
        }
    }

    /**
     * zmeni tlacitko zavrit na zrusit a naopak
     *
     * @param isClose true pokud tlacitko slouzi k zavreni
     */
    private void changeCloseButton(boolean isClose) {
        if (isClose) {
            closeDefibrilator.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.GLOBAL_CANCEL));
            closeDefibrilator.setOnAction(event -> cancelReloading());
        } else {
            closeDefibrilator.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.GLOBAL_CLOSE));
            closeDefibrilator.setOnAction(event -> hideDefibrilator());
        }
    }

    /**
     * zruseni probihane akce
     */
    private void cancelReloading() {
        analyseAnimation = stopAnimation(analyseAnimation);
        reloading = stopAnimation(reloading);

        analyse.setProgress(0);
        reload.setProgress(0);

        disableImpedanceButtons(false);
        disablePowerButtons(false);
        changeCloseButton(false);
        shock.setDisable(true);
        playSounds.stopShockReadyAnimation();

        if (isAED) {
            changeElectrods();
        } else {
            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_MENU_ENERGY));
        }
    }

    /**
     * zastavi pozadovanou animaci
     *
     * @param tl animace k zastaveni
     * @return null
     */
    private Timeline stopAnimation(Timeline tl) {
        if (tl != null) {
            tl.stop();
        }
        return null;
    }

    public boolean isAED() {
        return isAED;
    }

    /**
     * zobrazuje objekty v gridpane podle modu
     *
     * @param isAED         true pokud je automaticky mod
     * @param mode          nazev modu
     * @param manualSetting
     * @param AEDfunction
     */
    private void setModeDefibrilator(boolean isAED, String mode, GridPane manualSetting, GridPane AEDfunction) {
        typeOfDefibrilator.setText(mode);
        this.isAED = isAED;
        AEDfunction.setVisible(isAED);
    }

    /**
     * aktivni metoda pro spustenou analyzu - ovlada progress pro analyzu
     */
    private void analyseAnimation() {
        double progress = analyse.getProgress();
        analyse.setProgress(progress + 0.07);

        if (progress > 0.25 && analysaResult) {
            reloadDefibrilator(false);
        }

        if ((int) progress == 1 && !analysaResult) {
            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_DONT_SHOCK));
            playSounds.playSound(PlaySounds.DONT_GIVE_SHOCK);
        }

        if ((int) progress == 1) {
            analyse.setProgress(0);
            analyseAnimation = stopAnimation(analyseAnimation);
            isAnalysaComplete = true;
        }
    }

    /**
     * nastavuje defibrilator po kliknuti na checkbox s uvolnenou elektrodou
     *
     * @return true pokud je zakliknuto
     */
    private boolean setElectrodsAction() {
        if (!electrods) {
            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_FREE_ECG));
            playSounds.playSound(PlaySounds.FREE_EKG_ELECTROD);
            return false;
        } else {
            return true;
        }
    }

    /**
     * aktivni metoda pro nacitani energie pro vyboj v reloading timeline
     */
    private void reloadDefRun() {
        //nabijeni jen pro manualni
        if (((int) reload.getProgress()) < 1) {
            reload.setProgress(reload.getProgress() + 0.05);
            return;
        }

        if (!handleShockReady()) return;

        chargeWithAnalyse();
    }

    /**
     * overuje zda muze byt spusten vyboj ve funkci reloadDefRun
     *
     * @return true pokud neni manualni a je spravna impedance
     */
    private boolean handleShockReady() {
        //overeni spravne impedance
        if (this.impedance != 1) {
            infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_WRONG_IMP));
            return false;
        } else {
            reloading = stopAnimation(reloading);
            disableImpedanceButtons(true);
            shock.setDisable(false);

            if (!isAED) {
                infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_SHOCK_READY));
                playSounds.playSound(PlaySounds.GIVE_SHOCK);
                playSounds.startShockReadyAnimation();
                return false;
            }
            return true;
        }
    }

    /**
     * umozni podat naboj pokud probehla analyza i nabiti
     */
    private void chargeWithAnalyse() {
        //nabijeni synchronizovane s AED
        if (((int) reload.getProgress()) == 1 && isAnalysaComplete) {
            if (analysaResult) {
                infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_GIVE_SHOCK));
                playSounds.playSound(PlaySounds.GIVE_SHOCK);
                playSounds.startShockReadyAnimation();
            } else {
                infoLabel.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_DONT_SHOCK));
                playSounds.playSound(PlaySounds.DONT_GIVE_SHOCK);
            }
        }
    }

    public void pauseAnimations() {
        if (reloading != null) {
            reloading.pause();
        }

        if (playSounds != null) {
            playSounds.stopSound();
        }

        if (countingTimeLast != null) {
            countingTimeLast.pause();
        }

        if (countingTimeTotal != null) {
            countingTimeTotal.pause();
        }

        if (analyseAnimation != null) {
            analyseAnimation.pause();
        }

        playSounds.stopShockReadyAnimation();
    }

    public void resumeAnimation() {
        if (reloading != null) {
            reloading.play();
        }

        if (countingTimeLast != null) {
            countingTimeLast.play();
        }

        if (countingTimeTotal != null) {
            countingTimeTotal.play();
        }

        if (analyseAnimation != null) {
            analyseAnimation.play();
        }
    }

    public void clear() {
        if (reloading != null) {
            reloading.stop();
        }

        if (playSounds != null) {
            playSounds.stopSound();
            playSounds.stopShockReadyAnimation();
        }

        if (countingTimeLast != null) {
            countingTimeLast.stop();
        }

        if (countingTimeTotal != null) {
            countingTimeTotal.stop();
        }

        if (analyseAnimation != null) {
            analyseAnimation.stop();
        }

        reloading = null;
        playSounds = null;
        countingTimeLast = null;
        countingTimeTotal = null;
        analyseAnimation = null;
    }
}