package remoter_GUI;

import java.net.URL;
import java.util.*;

import data.ECGUtils;

import communication.AvailableMonitor;
import communication.Server;
import data.Utils;
import data.VitalData;
import gui_components.Alerts;
import gui_components.CreateAboutPanel;
import gui_components.GuiUtils;
import gui_components.Settings;
import javafx.animation.Animation;
import javafx.animation.KeyFrame;
import javafx.animation.Timeline;
import javafx.application.Platform;
import javafx.beans.property.Property;
import javafx.beans.property.SimpleListProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.concurrent.Task;
import javafx.fxml.FXML;
import javafx.fxml.Initializable;
import javafx.scene.control.*;
import javafx.scene.control.cell.PropertyValueFactory;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.GridPane;
import javafx.scene.paint.Color;
import javafx.scene.shape.Circle;
import javafx.stage.Stage;
import javafx.util.Duration;
import language.KeysForLanguage;
import language.LoadFXMLWithResourceBundle;
import settings.CreateGeneralPanel;

public class Controller implements Initializable {

    @FXML
    CheckMenuItem defibrilatorSendImmediately;

    @FXML
    GridPane columnWidthGridPane;
    @FXML
    TabPane userPanels;
    @FXML
    BorderPane avaibleMonitorsPane;

    @FXML
    Label heartRateLabel, ppLabel, sysPressureLabel, diaPressureLabel, respirationRateLabel, statusOfConnectionLabel,
            spo2Label, co2Label, timeLabel, temperatureLabel;

    @FXML
    Circle statusOfConnectionCircle;

    @FXML
    TextField monitorIpAddressField, portField;

    @FXML
    TableView<QuerySendData> queryTable;
    @FXML
    TableColumn nameOfCurveColumn, timeOfCurveColumn;

    @FXML
    ComboBox<String> impendaceCB, curveOfResurection;

    @FXML
    CheckBox successOfAnalysesChB, connectECGElectrodsChB, connectElectrodsForTerapyChB, successOfDefibrilationChB,
            hearthRateChB, PPChB, systolicPressureChB, diastolicPressureChB, respiratoryRateChB, SPO2ChB, ETCO2ChB,
            temperatureChB, connByIpCheckbox;

    @FXML
    ToggleButton minutes, seconds;
    @FXML
    Button addToQueryButton, newTaskButton, updateQueryButton, cancelUpdateQueryButton, findMonitorsButton, disconnectButton, sendButton,
            connectionButton;

    @FXML
    private ListView<String> graphList;
    @FXML
    private ListView<AvailableMonitor> avaibleMonitorsListView;

    private Timeline setTimeAnimation;
    private Property<ObservableList<QuerySendData>> queueDataToSend;

    private int timeConst;
    private Server server;
    private ResourceBundle langResource;
    private Settings settings;
    //private static final ResourceBundle langResource =  LoadFXMLWithResourceBundle.getLangResource();

    /**
     * Initialization all components in remoter before running application, and
     * also set necessary data.
     *
     * @param location  location.
     * @param resources resources.
     */
    @Override
    public void initialize(URL location, ResourceBundle resources) {
        VitalSlider.prop.set(columnWidthGridPane.widthProperty().doubleValue());
        this.langResource = resources;
        //Fill list with ECG curves
        setListOfGraphs();

        settings = new Settings();
        settings.initPanels(FXCollections.observableArrayList(new CreateGeneralPanel()));

        //Run thread, dispatching data in queue.
        timerRun();

        //Init. queue for data.
        queueDataToSend = new SimpleListProperty<>(FXCollections.observableArrayList());
        queryTable.itemsProperty().bindBidirectional(queueDataToSend);

        //Init. table with queue.
        settingUpQueueTable();

        timeConst = 1;

        cancelUpdateQueryButton.setOnAction(event -> cancelUpdateAction());
        updateQueryButton.setOnAction(event -> saveChangesQuery());

        changeStatusOfConnection(false);
        setToggleButtons();
        setDefibrillatorRetrieverData();
        monitorIpAddressField.setDisable(true);

        Properties properties = Utils.loadProperties(Utils.SETTINGS_PROP_PATH);
        portField.setText(properties.getProperty(Utils.PORT_PROP));
        monitorIpAddressField.setText(properties.getProperty(Utils.MONITOR_IP_ADDRESS));

        server = new Server();
    }

    /**
     * End application.
     */
    public void exitApplication() {
        Platform.exit();
    }

    /**
     * Fullscreen mode of application.
     */
    public void fullscreen() {
        Stage stage = (Stage) timeLabel.getScene().getWindow();
        stage.setFullScreen(true);
    }

    @FXML
    public void doSettings() {
        settings.showSettings();
    }

    @FXML
    public void doAbout() {
        new CreateAboutPanel(false);
    }

    /**
     * Get time from text label for timing.
     *
     * @return time value.
     */
    private int getTime() {
        return Integer.parseInt(timeLabel.getText());
    }

    /**
     * Set time in timing text label.
     *
     * @param newTime new time by set.
     */
    private void setTime(int newTime) {
        final int setNewTime = newTime >= 0 ? newTime > 999 ? 999 : newTime : 0;
        Platform.runLater(() -> timeLabel.setText(setNewTime + ""));
    }

    /**
     * Fill list with loaded ECG curves.
     */
    private void setListOfGraphs() {
        graphList.setItems(FXCollections.observableArrayList(ECGUtils.LoadECGCurvesMetadata(true)));
        graphList.getSelectionModel().selectedIndexProperty().addListener((observable, oldValue, newValue) ->
                VitalSlider.actualVitalData.setEcg(newValue.intValue()));
        graphList.getSelectionModel().select(0);

        curveOfResurection.getItems().addAll(graphList.getItems());
        curveOfResurection.getSelectionModel().select(0);
    }

    /**
     * Label with time label is animating while time increase/decrease button is pressed down.
     *
     * @param event mouse event.
     */
    @FXML
    public void startSetTimeAnimation(MouseEvent event) {
        setTimeAnimation = new Timeline(new KeyFrame(Duration.seconds(0.4), e -> {
            Button b = (Button) event.getSource();

            if (event.isPrimaryButtonDown()) {
                if (b.getText().compareTo("+") == 0) {
                    handlePlusTimeButton(event);
                } else {
                    handleMinusTimeButton(event);
                }
            }
        }));

        setTimeAnimation.setCycleCount(Animation.INDEFINITE);
        setTimeAnimation.play();
    }

    /**
     * Decreasing time.
     *
     * @param e mouse event.
     */
    @FXML
    public void handleMinusTimeButton(MouseEvent e) {
        setTime(getTime() - timeConst);
    }

    /**
     * Increasing time.
     *
     * @param e mouse event.
     */
    @FXML
    protected void handlePlusTimeButton(MouseEvent e) {
        setTime(getTime() + timeConst);
    }

    /**
     * Stop updating changes in time label
     */
    @FXML
    protected void stopSetTimeAnimation() {
        setTimeAnimation.stop();
        setTimeAnimation = null;
    }

    /**
     * upravi visibility vlastnost u tlacitek pro pridavani a upravu
     *
     * @param show true pokud chceme zobrazit upravovaci tlacitka
     */
    private void showUpdateItem(boolean show) {
        updateQueryButton.setVisible(show);
        cancelUpdateQueryButton.setVisible(show);
        addToQueryButton.setVisible(!show);
        newTaskButton.setVisible(!show);
    }

    /**
     * Handling send button
     */
    @FXML
    protected void handleSendButtonAction() {
        if (isSendNow()) {
            server.sendData(VitalSlider.prepareVitalData());
        } else {
            addToQueueToSendInFuture();
        }
    }

    /**
     * Get name of selected ECG curve in the list.
     *
     * @return ECG curve name.
     */
    private String getNameECGCurve() {
        int index = graphList.getSelectionModel().getSelectedIndex();
        return index == -1 ? "" : graphList.getItems().get(index);
    }

    /**
     * If data should be send immediately.
     *
     * @return true - send now.
     */
    private boolean isSendNow() {
        return getTime() == 0;
    }

    /**
     * Add data to queue.
     */
    @FXML
    public void addToQueueToSendInFuture() {
        QuerySendData dataToSend = new QuerySendData(getNameECGCurve(), getTime(), true,
                (VitalData) Utils.deepCopy(VitalSlider.actualVitalData));
        queueDataToSend.getValue().add(dataToSend);
        userPanels.getSelectionModel().select(2);
        setTime(0);
    }

    /**
     * Remove selected data from queue.
     */
    @FXML
    public void deleteFromQuery() {
        QuerySendData selectedData = queryTable.getSelectionModel().getSelectedItem();

        if (selectedData == null) return;

        VitalData queryData = selectedData.getQueryVitalData();
        queueDataToSend.getValue().remove(selectedData);

        setActualQueryData(null);
        showUpdateItem(queryData == null);
    }

    /**
     * Load data in the monitor labels and sliders according to selected item in queue.
     */
    @FXML
    private void loadValuesFromQueue() {
        QuerySendData selectedData = queryTable.getSelectionModel().getSelectedItem();

        if (selectedData == null) return;

        if (selectedData.getQueryVitalData() != null) {
            setActualQueryData(selectedData);
            showUpdateItem(true);
        } else {
            showUpdateItem(false);
        }

        userPanels.getSelectionModel().select(2);
    }

    private void setActualVitalData(VitalData vitalData) {
        vitalData = vitalData == null ? new VitalData() : vitalData;

        graphList.getSelectionModel().select(vitalData.getEcg());
        VitalSlider.actualVitalData.setVitalData(vitalData);
        successOfAnalysesChB.setSelected(vitalData.getSuccessOfAnalyse());
        successOfDefibrilationChB.setSelected(vitalData.getSuccessOfDefibrilation());
        impendaceCB.getSelectionModel().select(vitalData.getImpedance());
    }

    private void setActualQueryData(QuerySendData data) {
        data = data == null ? new QuerySendData() : data;

        setActualVitalData(data.getQueryVitalData());
        setTime(data.getTime());
    }

    /**
     * Save changes into selected data in queue.
     */
    private void saveChangesQuery() {
        QuerySendData selectedData = queryTable.getSelectionModel().getSelectedItem();

        if (selectedData != null) {
            selectedData.setQueryVitalData((VitalData) Utils.deepCopy(VitalSlider.actualVitalData));
            selectedData.setTime(getTime());

            queryTable.refresh();
            queryTable.getSelectionModel().clearSelection();

            showUpdateItem(false);
        }
    }

    /**
     * Create new task for stop dispatching queue.
     */
    @FXML
    public void stopSendingData() {
        QuerySendData data = new QuerySendData("STOP", 0, false, null);
        queueDataToSend.getValue().add(data);
    }

    /**
     * Run thread animation for dispatching data.
     */
    private void timerRun() {
        Timeline t1 = new Timeline(new KeyFrame(Duration.seconds(1), e -> sendDataPrepare()));
        t1.setCycleCount(Animation.INDEFINITE);
        t1.play();
    }

    @FXML
    public void connectToMonitor() {
        String ipAddress = null;
        if (connByIpCheckbox.isSelected()) {
            ipAddress = monitorIpAddressField.getText();
            if (ipAddress != null && !ipAddress.isEmpty()){
                Utils.updateProperties(Utils.MONITOR_IP_ADDRESS, ipAddress, Utils.SETTINGS_PROP_PATH);
            } else {
                Alerts.warningAlert(langResource.getString(KeysForLanguage.REMOTER_ERROR_COMM), langResource.getString(KeysForLanguage.REMOTER_NOT_SEL_MONITOR), langResource.getString(KeysForLanguage.REMOTER_SET_MONITOR));
            }
        } else {
            AvailableMonitor availableMonitor = avaibleMonitorsListView.getSelectionModel().getSelectedItem();
            if (availableMonitor != null) {
                ipAddress = availableMonitor.getIpAddress();
            } else {
                Alerts.warningAlert(langResource.getString(KeysForLanguage.REMOTER_ERROR_COMM), langResource.getString(KeysForLanguage.REMOTER_NOT_SEL_MONITOR), langResource.getString(KeysForLanguage.REMOTER_SET_MONITOR));
                return;
            }
        }

        if (ipAddress != null && GuiUtils.checkIpAddress(ipAddress) && GuiUtils.checkPort(portField.getText())) {
            Utils.updateProperties(Utils.PORT_PROP, portField.getText(), Utils.SETTINGS_PROP_PATH);
            final String finIpAddress = ipAddress;
            Task<Void> task = new Task<Void>() {
                @Override
                protected Void call() throws Exception {
                    int port = Integer.valueOf(portField.getText());
                    boolean isConnected = server.connectToMonitor(finIpAddress, port);
                    if (isConnected) {
                        Platform.runLater(() -> {
                            changeStatusOfConnection(isConnected);
                        });
                        server.waitForDisconnect(port);

                        Platform.runLater(() -> {
                            changeStatusOfConnection(false);
                            avaibleMonitorsListView.getItems().clear();
                            Alerts.infoAlert(langResource.getString(KeysForLanguage.REMOTER_COMM), langResource.getString(KeysForLanguage.REMOTER_MONITOR_DISC), "");
                        });

                    } else {
                        Platform.runLater(() -> {
                            Alerts.warningAlert(langResource.getString(KeysForLanguage.REMOTER_ERROR_COMM), langResource.getString(KeysForLanguage.REMOTER_IP_MONITOR) + " " + finIpAddress + " " + langResource.getString(KeysForLanguage.REMOTER_NOT_FOUND), "");
                        });
                    }
                    return null;
                }
            };
            new Thread(task).start();
        }

    }

    /**
     * metoda pro znovu navazani kontaktu s ovladacem
     */
    @FXML
    public void findMonitors() {
        findMonitorsButton.setText(langResource.getString("remote.connection.findMonitorsButton.searching"));
        if (!GuiUtils.checkPort(portField.getText())) {
            return;
        } else {
            Utils.updateProperties(Utils.PORT_PROP, portField.getText(), Utils.SETTINGS_PROP_PATH);
        }
        Task<Void> task = new Task<Void>() {
            @Override
            protected Void call() throws Exception {
                List<AvailableMonitor> monitors = server.getAvaibleMonitors(Integer.valueOf(portField.getText()));
                ObservableList<AvailableMonitor> obsListMonitors = FXCollections.observableArrayList();
                obsListMonitors.addAll(monitors);
                Platform.runLater(() -> {
                    avaibleMonitorsListView.setItems(obsListMonitors);
                    avaibleMonitorsListView.getSelectionModel().select(0);
                    findMonitorsButton.setText(langResource.getString("remote.connection.findMonitorsButton"));
                });
                return null;
            }
        };
        new Thread(task).start();
    }

    @FXML
    public void disconnect() {
        server.disconnectMonitor();
    }

    @FXML
    public void checkConnByIp() {
        boolean connByIp = connByIpCheckbox.isSelected();
        monitorIpAddressField.setDisable(!connByIp);
        avaibleMonitorsListView.setDisable(connByIp);
        findMonitorsButton.setDisable(connByIp);

    }


    /**
     * This method is called in thread created as animation which checks
     * periodically data, if they are ready for dispatching to the monitor.
     */
    private void sendDataPrepare() {
        if (queueDataToSend.getValue().isEmpty()) return;

        QuerySendData headData = queueDataToSend.getValue().get(0);
        if (!headData.getSend()) { //If data is set as not dispatching to the monitor (like a stop marker).
            queueDataToSend.getValue().remove(0);
        } else {
            headData.incTime();

            if (headData.getTime() <= 0) {
                server.sendData(VitalSlider.prepareVitalData(headData.getQueryVitalData()));
                queueDataToSend.getValue().remove(0);
            }

            Platform.runLater(() -> queryTable.refresh());
        }
    }

    /**
     * nastaveni statusu pripojeni a vse s tim souvisejici
     */
    public void changeStatusOfConnection(boolean isConnected) {
        Platform.runLater(() -> {
            findMonitorsButton.setVisible(!isConnected);
            disconnectButton.setVisible(isConnected);
            userPanels.setVisible(isConnected);
            avaibleMonitorsPane.setVisible(!isConnected);


            Color color = isConnected ? Color.GREEN : Color.RED;
            String connectioInfoText = isConnected ? langResource.getString(KeysForLanguage.REMOTER_MONITOR_CONNECTED) : langResource.getString(KeysForLanguage.REMOTER_MONITOR_DISC);
            findMonitorsButton.setText(langResource.getString("remote.connection.findMonitorsButton"));


            statusOfConnectionCircle.setFill(color);
            statusOfConnectionLabel.setText(connectioInfoText);
            statusOfConnectionLabel.setTextFill(color);
            sendButton.setDisable(!isConnected);
        });
    }

    /**
     * zrusi upravy provadene na vybrane polozce z fronty
     */
    private void cancelUpdateAction() {
        queryTable.getSelectionModel().clearSelection();
        setActualQueryData(null);
        showUpdateItem(false);
    }

    /**
     * Set table with data queue table.
     */
    private void settingUpQueueTable() {
        //Responsive columns.
        queryTable.setColumnResizePolicy(TableView.CONSTRAINED_RESIZE_POLICY);
        nameOfCurveColumn.prefWidthProperty().bind(queryTable.widthProperty().multiply(0.8));
        timeOfCurveColumn.prefWidthProperty().bind(queryTable.widthProperty().multiply(0.2));

        //Handle mouse action on item in queue table.
        queryTable.setOnMouseClicked(event1 -> loadValuesFromQueue());

        //Binding columns with names of their cell.
        nameOfCurveColumn.setCellValueFactory(new PropertyValueFactory<>("name"));
        timeOfCurveColumn.setCellValueFactory(new PropertyValueFactory<>("time"));
    }

    private void setToggleButtons() {
        ToggleGroup toggleGroup = new ToggleGroup();
        minutes.setToggleGroup(toggleGroup);
        seconds.setToggleGroup(toggleGroup);
        seconds.toFront();

        toggleGroup.selectedToggleProperty().addListener((observable, oldValue, newValue) -> {
            if (newValue == null) {
                oldValue.setSelected(true);
            }

            ToggleButton chosenButton = (ToggleButton) toggleGroup.getSelectedToggle();
            chosenButton.toFront();

            timeConst = chosenButton.equals(minutes) ? 60 : 1;
        });
    }

    private void setDefibrillatorRetrieverData() {
        connectECGElectrodsChB.setOnAction(event -> {
            if (connectECGElectrodsChB.isSelected()) {
                connectElectrodsForTerapyChB.setDisable(false);
            } else {
                connectElectrodsForTerapyChB.setDisable(true);
                connectElectrodsForTerapyChB.setSelected(false);
            }

            VitalSlider.actualVitalData.setConnectECG(connectECGElectrodsChB.isSelected());
            sendDefibrillatorDataNow();
        });

        connectElectrodsForTerapyChB.setOnAction(event -> {
            VitalSlider.actualVitalData.setConnectTherapy(connectElectrodsForTerapyChB.isSelected());
            sendDefibrillatorDataNow();
        });


        impendaceCB.getItems().setAll(FXCollections.observableArrayList(
                LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.MONITOR_DEFIBRILATOR_IMPEDANCE_LOW),
                LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.MONITOR_DEFIBRILATOR_IMPEDANCE_MEDIUM),
                LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.MONITOR_DEFIBRILATOR_IMPEDANCE_HIGH)
        ));
        impendaceCB.getSelectionModel().select(1);
        impendaceCB.valueProperty().addListener((observable, oldValue, newValue) -> {
            if (newValue != null) {
                VitalSlider.actualVitalData.setImpedance(impendaceCB.getSelectionModel().getSelectedIndex());
                sendDefibrillatorDataNow();
            }
        });

        successOfAnalysesChB.setOnAction(event -> {
            VitalSlider.actualVitalData.setSuccessOfAnalyse(successOfAnalysesChB.isSelected());
            sendDefibrillatorDataNow();
        });

        successOfDefibrilationChB.setOnAction(event -> {
            VitalSlider.actualVitalData.setSuccessOfDefibrillation(successOfDefibrilationChB.isSelected());
            sendDefibrillatorDataNow();

            curveOfResurection.setDisable(!successOfDefibrilationChB.isSelected());
        });

        curveOfResurection.setOnMouseClicked(event -> {
            VitalSlider.actualVitalData.setResuscitationEcg(curveOfResurection.getSelectionModel().getSelectedIndex());
            sendDefibrillatorDataNow();
        });
    }

    private void sendDefibrillatorDataNow() {
        if (defibrilatorSendImmediately.isSelected()) {
            VitalSlider.actualVitalData.setChangeOnlyDefibrillator(true);
            server.sendData(VitalSlider.prepareVitalData());
            userPanels.getSelectionModel().select(1);
            VitalSlider.actualVitalData.setChangeOnlyDefibrillator(false);
        }
    }

    public void closeRemoter() {
        server.closeServer();
    }
}
