package vital_ecg;

import static java.util.concurrent.TimeUnit.NANOSECONDS;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;

import data.ECGConst;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;

/**
 * Trida, ktera zajistuje vypocty bodu pro danou krivku.
 *
 * @author Patrik Patera
 * @version 1.00
 */
public class ECGCalculation {

    /**
     * Data, ktera obsahuji navzorkovany komplex - typ krivky.
     */
    private ObservableList<ECGPoint> data;

    /**
     * Data, ktera obsahuji navzorkovany komplex - typ krivky.
     */
    private ObservableList<ObservableList<ECGPoint>> dataCurve;

    /**
     * Vlakno urcene pro vypocet bodu.
     */
    private Runnable calculation;

    /**
     * Planovac pro vypocty.
     */
    private ScheduledFuture<?> calculationTask;

    /**
     * Pozice bodu, ktery se ma vykreslovat z listu Data.
     */
    private int pos;

    /**
     * Maximalni pozice bodu, ktery se muze vykreslit z listu Data.
     */
    private int max_pos;

    /**
     * Tepova frekvence srdce.
     */
    private double HF;

    /**
     * Urcuje cas, ktery se ma cekat pri animaci ve sleep.
     */
    private double wait;

    /**
     * Priznak, ktery urcuje, jestli se ma vykreslovat krivka. Jinak konstantni
     * f-ce: y = ECGConst.Y_ECG1_VALUE.
     */
    private boolean plotCurve;

    /**
     * Priznak, jestli se ma vykreslovat z plotData pri true, nebo plotData2 pri
     * false.
     */
    private boolean plottingData1;

    /**
     * Znaci, kde v listu bodu je prechod mezi komplexem a pauzou.
     */
    private int markerInList;

    /**
     * Hodnota X-ove souradnice.
     */
    private float valueX;

    /**
     * Jestli se ma provest prepocet hodnot.
     */
    private boolean recalculate;

    /**
     * Priznak, jestli se prehrap pipnuti.
     */
    private boolean beep;

    /**
     * Planovac pro vypocet.
     */
    private final ScheduledExecutorService executor;

    /**
     * Prostrednik, zajistujici vzajemny prenos dat mezi kalkulaci a
     * vykreslovanim.
     */
    private final ECGData ecg;

    /**
     * Konstruktor, ktery vytvori tridy pro kalkulaci bodu k vykreslovani.
     *
     * @param executor manager pro vlakno.
     * @param ecg prostednik ECGData.
     */
    public ECGCalculation(ScheduledExecutorService executor, ECGData ecg) {
        this.plotCurve = true;

        this.ecg = ecg;
        this.executor = executor;

        initValues();
        recalculateValues();
        setCalculating();

        ecg.setWait(wait);
    }

    /**
     * Nastaveni pocatecnich hodnot pro vykreslovani krivky. Vzdy nutno volat
     * pred samotnym nastavenim hodnot dle dane krivky !!! Tedy pred metodou
     * recalculateValues();
     */
    private void initValues() {
        this.HF = 60;
        this.pos = 0;
        this.max_pos = 0;
        this.markerInList = 0;
        this.recalculate = false;
        this.plottingData1 = true;
        this.plotCurve = false;
        ecg.notConnected();
        notConnected();
    }

    /**
     * Nastavuje data s body tak, aby se vykreslovala prerusovana cara, ktera ma
     * za ukol indikovat, ze nejsou pripojeny elektrody na pacientovi.
     */
    public void notConnected() {
        // this.data = ECGConst.LINE;
        this.plotCurve = false;
    }

    /**
     * Vakno, ktere pocita body pro graf.
     */
    private void setCalculating() {
        calculation = () -> {
            if (ecg.getCountPoints() < ECGConst.MAX_COUNT_QUEUE) {
                if (valueX < ECGConst.DURATION_SECONDS) {
                    if ((pos >= markerInList && recalculate)) {
                        recalculate = false;

                        recalculateValues();
                        ecg.changeWait(wait);

                        pos = 0;
                    }

                    if (pos >= max_pos || pos == 0) {
                        pos = 0;
                        valueX += data.get(pos).getXValue();
                    } else {
                        valueX += (data.get(pos).getXValue() - data.get(pos - 1).getXValue());
                        beep = data.get(pos).getPlay();
                    }

                    ecg.sendData(new ECGPoint(valueX, data.get(pos).getECG1YValue(),
                            data.get(pos).getECG2YValue(), plottingData1, beep));

                    ecg.incPoint();
                    pos++;
                } else {
                    plottingData1 = !plottingData1;
                    valueX = 0;
                }
            }
        };
    }

    /**
     * Vypocita celkovou pauzu vsech nactenych vzorku.
     *
     * @return celkova pauza.
     */
    private double getTotalLengthPause() {
        double count_points = HF / ECGConst.COUNT_COMPLEX_IN_MINUTE;
        double length = 0;

        if (dataCurve != null) {
            for (ObservableList<ECGPoint> ob : dataCurve) {
                double tmpCalc = ECGConst.DURATION_SECONDS - count_points
                        * ob.get(ob.size() - 1).getXValue();
                length += tmpCalc / count_points;
            }
        }

        return length;
    }

    /**
     * Vypocteni dulezitych hodnot pro vykreslovani.
     */
    private void recalculateValues() {
        double totalLengthPause = getTotalLengthPause();
        double valueXPause;

        if (plotCurve) {
            data = FXCollections.observableArrayList();

            double count_points = HF / ECGConst.COUNT_COMPLEX_IN_MINUTE;
            double dec = count_points % ((int) count_points);
            double plus = count_points >= 1 ? dec * totalLengthPause : 0;
            float lastX = 0;

            max_pos = dataCurve.get(0).size() - 1;

            valueXPause = getValueXPause();

            for (ObservableList<ECGPoint> curve : dataCurve) {
                double pause = ECGConst.DURATION_SECONDS - count_points
                        * curve.get(curve.size() - 1).getXValue();
                pause /= count_points;

                for (ECGPoint point : curve) {
                    if (point.getXValue() < 0) {
                        data.add(new ECGPoint(-point.getXValue() + lastX,
                                point.getECG1YValue(), point.getECG2YValue(), true));
                    } else {
                        data.add(new ECGPoint(point.getXValue() + lastX,
                                point.getECG1YValue(), point.getECG2YValue()));
                    }
                }

                setPauseToList(pause, valueXPause);

                lastX = data.get(max_pos).getXValue();
            }

            if (totalLengthPause > 0) {
                this.wait = (data.get(max_pos).getXValue() + Math.abs(totalLengthPause) + plus)
                        / (data.size() + totalLengthPause / valueXPause + plus / valueXPause)
                        * ECGConst.SEC_TO_NANOSEC;
            } else {
                this.wait = (data.get(max_pos).getXValue() + totalLengthPause)
                        / (data.size()) * ECGConst.SEC_TO_NANOSEC;
            }

        } else {
            //TODO MOVE TO DEFINE ASYSTOLIA
            this.data = FXCollections.observableArrayList(new ECGPoint(0.01f, ECGConst.Y_ECG1_VALUE,
                    ECGConst.Y_ECG2_VALUE));
            max_pos = data.size() - 1;

            this.wait = ECGConst.DURATION_SECONDS / (ECGConst.DURATION_SECONDS
                    / data.get(0).getXValue()) * ECGConst.SEC_TO_NANOSEC;
            markerInList = 0;
            valueXPause = data.get(0).getXValue();
        }
    }

    /**
     * Nastaveni tepove frekvence srdce.
     *
     * @param value hodnota tepove frekvence.
     */
    public void setHR(int value) {
        /* Pri nastaveni jine tepove frekvence je nutno prepocitat hodnoty */
        System.out.println("Set new HearthRate value: " + value);
        this.HF = value;
        this.recalculate = true;
    }

    /**
     * Nastaveni nove krivky, ktera se ma vykreslovat v podobe seznamu
     * navzorkovanych dat.
     *
     * @param data nova navzorkovana data krivky.
     */
    public void setECGData(ObservableList<ObservableList<ECGPoint>> data) {
        /* Nastaveni nove krivky do data a prepocitani body */
        System.out.println("Settings new ECG DATA!");
        dataCurve = data;
        System.out.println("\tnova krivka predana");

        this.plotCurve = true;
        this.recalculate = true;
    }

    /**
     * Vypocet inkrementace behem pauzy tak, aby vykreslovani bylo plynule.
     * Prumerna X-ova inkrementace.
     *
     * @return inkrementace vykreslovani pauzy pri kazdem wait.
     */
    private float getValueXPause() {
        float sum = 0;
        int i, length = dataCurve.get(0).size();

        for (i = 1; i < length; i++) {
            sum += (dataCurve.get(0).get(i).getXValue() - dataCurve.get(0).get(i - 1).getXValue());
        }

        sum /= i;

        return sum;
    }

    /**
     * Prida body pauzy (mezi jednotlivymi komplexy) do listu s body dane
     * vykreslujici se krivky a oznaci markerem, kde konci komplex a zacina
     * pauza.
     */
    private void setPauseToList(double pauseLength, double valueXPause) {
        float countXpause = (float) valueXPause;
        max_pos = data.size() - 1;
        float lastXVvalue = this.data.get(max_pos).getXValue();
        float valueYECG1 = this.data.get(0).getECG1YValue();
        float valueYECG2 = this.data.get(0).getECG2YValue();

        this.markerInList = this.max_pos;

        for (float incX = lastXVvalue + countXpause; incX
                <= (pauseLength + lastXVvalue); incX += countXpause) {
            this.data.add(new ECGPoint(incX, valueYECG1, valueYECG2));
        }

        max_pos = data.size() - 1;
    }

    /**
     * Postara se o uzavreni a zaniku vsech vlaken vykreslovani a vypoctu.
     */
    public void clear() {
        executor.shutdownNow();
    }

    public ScheduledFuture<?> getCalculation() {
        return calculationTask;
    }

    public void setNewCalculation() {
        calculationTask = executor.scheduleAtFixedRate(calculation, 0, (long) wait, NANOSECONDS);
    }
}
