package vital_monitor;

import data.ECGUtils;
import data.Utils;
import data.VitalData;
import gui_components.*;
import javafx.animation.Animation;
import javafx.animation.KeyFrame;
import javafx.animation.Timeline;
import javafx.application.Platform;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.concurrent.Task;
import javafx.event.ActionEvent;
import javafx.fxml.FXML;
import javafx.fxml.Initializable;
import javafx.geometry.HPos;
import javafx.geometry.Insets;
import javafx.geometry.VPos;
import javafx.scene.chart.Chart;
import javafx.scene.chart.XYChart;
import javafx.scene.control.*;
import javafx.scene.image.Image;
import javafx.scene.layout.*;
import javafx.scene.text.Font;
import javafx.stage.Stage;
import javafx.util.Duration;
import javafx.util.Pair;
import language.KeysForLanguage;
import language.LoadFXMLWithResourceBundle;
import vital_ecg.ECGData;
import vital_ecg.ECGPoint;
import vital_monitor.settings.CreateBoundariesPanel;
import vital_monitor.settings.CreateGeneralPanel;
import vital_monitor.settings.CreateSoundPanel;
import vital_monitor.settings.ValuesAboutStateOfApplication;


import java.net.BindException;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.List;
import java.util.Properties;
import java.util.ResourceBundle;

import communication.Client;

public class Controller implements Initializable {

    @FXML
    Button aedButton, changeMode, closeNIBP, start, closeDefibrilator, NIBPButton, energy50, energy100, energy150,
            energy200, soundDefibrilator, soundNIBP, overViewButton, powerOnButton, manualButton, soundECGButton,
            shockButton, connectionButton, withoutConnectionButton, stopWaitingConnButton, analyzeButton;
    @FXML
    ProgressBar reload, NIBPprogress, analysaProgress, poweringOnProgress;
    @FXML
    Label powerinOnLabel, NIBPmodeLabel, NIBPmodeValueLabel, NIBPpressureLabel, NIBPpressure2Label, power,
            timeTotalLabel, timeLastLabel, numberOfShocksLabel, typeOfDefibrilator, NIBPtime, infoDefibrilatorLabel,
            ipAddressLabel, portLabel, monitorNameLabel, connectionInfoLabel;
    @FXML
    GridPane defibrilator, firstGraph, secondGraph, thirdGraph, valuesFromRemoter, powerOn,
            NIBPGridPane, NIBPOverview, AEDfunction, sideMenuPane, MenuDefibrilator, manualSetting, impedancePane,
            buttonsEnergy, siideButtonsPane, connectionPane;
    @FXML
    MenuItem pauseAplication, resumeAplication, shutDownMonitor, aboutMenu;
    @FXML
    CheckMenuItem graph1, graph2;
    @FXML
    CheckBox connectEcgElectrods, connectTherapyElectrods;
    @FXML
    GridPane logoPane;
    @FXML
    TextField monitorNameField, portField;
    @FXML
    MenuItem reconnectMonitor;
    @FXML
    NIBPVitalLabel nibpLabel;
    @FXML
    ColumnConstraints poweOnLabelConst;

    private ECGData ecgData;
    private NIBPHandler NIBPhandler;
    private DefibrilatorHandle defibrilatorHandle;

    private Settings settings;
    private static Client client;
    private static Thread connectionThread;

    /**
     * animace casu pro defibrilator
     */
    private Timeline poweringOnAnimation;
    private boolean ecgSound;

    private static List<Pair<String, List<float[][]>>> loadedECGCurves = ECGUtils.loadECGData();
    private static final ResourceBundle langResource =  LoadFXMLWithResourceBundle.getLangResource();

    /**
     * spusteno pred vykreslenim monitoru, vytvari jednotlive handlery funkci a
     * nastavuje potrebne bindingy k monitoru
     *
     * @param location  location.
     * @param resources resources.
     */
    @Override
    public void initialize(URL location, ResourceBundle resources) {
        //inicializace objektu trid
        defibrilatorHandle = new DefibrilatorHandle(defibrilator, typeOfDefibrilator, power, numberOfShocksLabel,
                timeLastLabel, timeTotalLabel, reload, shockButton, infoDefibrilatorLabel, analysaProgress, closeDefibrilator,
                impedancePane, buttonsEnergy);

        NIBPhandler = new NIBPHandler(NIBPprogress, NIBPmodeLabel, NIBPmodeValueLabel, NIBPpressureLabel, NIBPpressure2Label,
                changeMode, start, NIBPGridPane, NIBPtime, NIBPOverview, closeNIBP, nibpLabel);

        defibrilatorHandle.setImpedanceFromRemoter(1);
        defibrilatorHandle.setActivePowerButton(energy200);

        setVisibilityAllComponents(false);

        reload.minWidthProperty().bind(((GridPane) reload.getParent()).heightProperty().subtract(100));
        analysaProgress.minWidthProperty().bind(((GridPane) analysaProgress.getParent()).widthProperty().subtract(70));
        setElectrodsCheckboxes();

        setSoundButtonDefibrilator();
        setSoundNIBPButton();
        ecgSound = true;
        setVisibilityAllComponents(false);

        soundECGButton = new Button();

        settings = new Settings();
        settings.initPanels( FXCollections.observableArrayList(new CreateBoundariesPanel(valuesFromRemoter),
                new CreateGeneralPanel(), new CreateSoundPanel(soundDefibrilator, soundNIBP, soundECGButton)));

        FXCollections.observableArrayList(new CreateBoundariesPanel(valuesFromRemoter),
                new CreateGeneralPanel(), new CreateSoundPanel(soundDefibrilator, soundNIBP, soundECGButton));

        Properties properties = Utils.loadProperties(Utils.SETTINGS_PROP_PATH);
        portField.setText(properties.getProperty(Utils.PORT_PROP));
        monitorNameField.setText(properties.getProperty(Utils.MONITOR_NAME_PROP));

        try {
            ipAddressLabel.setText(langResource.getString(KeysForLanguage.MONITOR_CONN_IP)+": " + InetAddress.getLocalHost().getHostAddress() + ")");
        } catch (UnknownHostException e) {
            e.printStackTrace();
        }

        shockButton.setDisable(true);

        sideMenuPane.widthProperty().addListener((observable, oldValue, newValue) -> {
            if (newValue != null) {
                poweOnLabelConst.setMinWidth(newValue.doubleValue());
                poweOnLabelConst.setMaxWidth(newValue.doubleValue());
            }
        });
    }

    @FXML
    public void defibrilatorAEDHandle() {
        if (ValuesAboutStateOfApplication.isDefibrilatorAEDOn) return;
        if (ValuesAboutStateOfApplication.isDefibrilatorManualOn) hideDefibrilator();

        normalMonitorStart(true);
    }

    @FXML
    public void defibrilatorManualHandle() {
        if (ValuesAboutStateOfApplication.isDefibrilatorManualOn) return;
        if (ValuesAboutStateOfApplication.isDefibrilatorAEDOn) hideDefibrilator();
        if (ValuesAboutStateOfApplication.isNIBPON) hidePressure();

        normalMonitorStart(false);
    }

    @FXML
    public void nibpHandle() {
        if (ValuesAboutStateOfApplication.isDefibrilatorAEDOn || ValuesAboutStateOfApplication.isDefibrilatorManualOn) {
            hideDefibrilator();
        }

        showPressure();
    }

    @FXML
    public void waitForMonitor() {
        String monitorName = monitorNameField.getText();

        String portStr = portField.getText();
        if (!GuiUtils.checkMonitorName(monitorName) || !GuiUtils.checkPort(portStr)) {
            return; //invalid configuration
        }

        Utils.updateProperties(Utils.MONITOR_NAME_PROP, monitorName, Utils.SETTINGS_PROP_PATH);
        Utils.updateProperties(Utils.PORT_PROP, portStr, Utils.SETTINGS_PROP_PATH);

        if (client != null) client.setMonitClosed(true);

        client = new Client(this, monitorName, Integer.valueOf(portStr));
        Task<Void> task = new Task<Void>() {
            @Override
            protected Void call() throws Exception {
                Platform.runLater(() -> changeConnectionPane(true));

                String remoterIpAddress = null;
                try {
                    remoterIpAddress = client.getRemoterIpAddress(monitorName);
                } catch (BindException e) {
                    Platform.runLater(() -> Alerts.warningAlert(langResource.getString(KeysForLanguage.MONITOR_CONN_ERROR), "Port "
                            + portStr + langResource.getString(KeysForLanguage.MONITOR_CONN_PORT_RESERVED), langResource.getString(KeysForLanguage.MONITOR_CONN_PORT_SET)));
                } catch (SocketException e) {
                    Platform.runLater(() -> Alerts.warningAlert(langResource.getString(KeysForLanguage.MONITOR_CONN_ERROR), e.getLocalizedMessage(),
                            ""));
                }

                if (remoterIpAddress != null) {
                    Platform.runLater(() -> showOnlySideButtons(true));

                    client.startListening(remoterIpAddress);

                    Platform.runLater(() -> {
                        if (ecgData != null) {
                            shutDownMonitor();
                        }
                    });
                }
                Platform.runLater(() -> {
                    showOnlySideButtons(false);
                    changeConnectionPane(false);
                    Alerts.infoAlert(langResource.getString(KeysForLanguage.MONITOR_CONN_COMMUN), langResource.getString(KeysForLanguage.MONITOR_CONN_DISCONNECT), "");
                });
                return null;
            }
        };

        if (connectionThread != null) connectionThread.interrupt();

        connectionThread = new Thread(task);
        connectionThread.start();
    }

    private void changeConnectionPane(boolean waiting) {
        connectionButton.setVisible(!waiting);
        withoutConnectionButton.setVisible(!waiting);
        portField.setDisable(waiting);
        monitorNameField.setDisable(waiting);
        portLabel.setDisable(waiting);
        monitorNameLabel.setDisable(waiting);
        stopWaitingConnButton.setVisible(waiting);
        String connInfoText = waiting ? langResource.getString(KeysForLanguage.MONITOR_CONN_WAITING) : langResource.getString(KeysForLanguage.MONITOR_CONN_CONNECT);
        connectionInfoLabel.setText(connInfoText);
    }

    private void showOnlySideButtons(boolean show) {
        reconnectMonitor.setDisable(!show);
        siideButtonsPane.setVisible(show);
        connectionPane.setVisible(!show);
        powerOnButton.setVisible(show);
        connectionInfoLabel.setVisible(!show);
    }

    @FXML
    public void stopMonitorCommunication() {
        client.setMonitClosed(true);
    }

    @FXML
    public void withoutConnection() {
        showOnlySideButtons(true);
    }

    @FXML
    public void reconnectMonitor() {
        if (connectionThread != null) connectionThread.interrupt();

        if (client != null) {
            client.setMonitClosed(true);
        }

        if (getECGData() != null) {
            getECGData().clear();
        }

        VitalLabel.vitalData = new VitalData();

        showOnlySideButtons(false);
        stopMonitor();
    }

    private ECGData getECGData() {
        return ecgData;
    }

    /**
     * prida graf na prvni misto v gridpanu monitoru
     *
     * @param nameOfGraph jmeno zobrazovaneho grafu
     * @param graph       graf ktery se ma zobrazit
     */
    private void addGraph(String nameOfGraph, XYChart graph, int position) {
        graph.setPadding(new Insets(0, 0, 20, 0));

        firstGraph.add(createPanelWithButton(new Label(nameOfGraph), soundECGButton), 0, position);
        firstGraph.add(graph, 1, position);
    }

    /**
     * Prida postrani panel vedle grafu
     *
     * @param name
     * @return Vyrvoreni panel.
     */
    private GridPane createPanelWithButton(Label name, Button button) {
        GridPane gridPane = new GridPane();
        name.setFont(Font.font("System", 20));
        name.setPadding(new Insets(3, 0, 0, 0));

        ColumnConstraints columnConstraints = new ColumnConstraints();
        columnConstraints.setMinWidth(Region.USE_COMPUTED_SIZE + 80);

        RowConstraints rowConstraints1 = new RowConstraints(50);
        rowConstraints1.setMaxHeight(100);
        rowConstraints1.setValignment(VPos.TOP);

        gridPane.getColumnConstraints().addAll(columnConstraints);
        gridPane.getRowConstraints().addAll(rowConstraints1);

        gridPane.add(name, 0, 0);
        gridPane.add(button, 0, 1);

        gridPane.getColumnConstraints().get(0).setHalignment(HPos.CENTER);
        button.setMinWidth(50);
        button.setMinHeight(40);
        gridPane.setMargin(button, new Insets(0, 0, 35, 12));
        gridPane.setMargin(name, new Insets(0, 0, 0, 10));
        button.setStyle("-fx-cursor: hand;");

        setBackgroundImageOnSoundButton(button, false);

        // Zvuk pro ECG graf
        button.setOnAction(event -> {
            setBackgroundImageOnSoundButton(button, ecgSound);
            ecgSound = !ecgSound;
            ecgData.setBeep(ecgSound);
        });

        return gridPane;
    }

    private void setBackgroundImageOnSoundButton(Button button, boolean isActive) {
        String imagePath = isActive ? "images/disabledSound.jpg" : "images/activeSound.jpg";

        BackgroundImage backgroundImage = new BackgroundImage(
                new Image(getClass().getResource(imagePath).toExternalForm()),
                BackgroundRepeat.NO_REPEAT,
                BackgroundRepeat.NO_REPEAT,
                BackgroundPosition.CENTER,
                BackgroundSize.DEFAULT
        );

        Background background = new Background(backgroundImage);
        button.setBackground(background);
    }

    private void createNormalGraphPane(GridPane graphGrid, String nameOfGraph, Chart graph) {
        graph.setPadding(new Insets(0, 0, 20, 0));
        Label name = new Label(nameOfGraph);
        name.setFont(Font.font("System", 20));
        name.setPadding(new Insets(3, 0, 0, 0));

        graphGrid.getColumnConstraints().get(0).setHalignment(HPos.CENTER);
        graphGrid.getRowConstraints().get(0).setValignment(VPos.TOP);
        graphGrid.add(name, 0, 0);
        graphGrid.add(graph, 1, 0);
        graphGrid.setMargin(name, new Insets(0, 0, 0, 10));
    }

    private void removeGraph(int number) {
        switch (number) {
            case 1:
                firstGraph.getChildren().clear();
                break;
            case 2:
                secondGraph.getChildren().clear();
                break;
            case 3:
                thirdGraph.getChildren().clear();
                break;
        }
    }

    /*
     ********************************************************************************************************************
     *                                           METODY PRO KLAVESOVE ZKRATKY
     ********************************************************************************************************************
     */

    /**
     * vypne aplikaci
     */
    @FXML
    public void exitApplication() {
        Platform.exit();
    }

    /**
     * zobrazi aplikaci ve fullscreen modu
     */
    @FXML
    public void fullscreen() {
        Stage stage = ((Stage) valuesFromRemoter.getScene().getWindow());
        stage.setFullScreen(true);
    }

    @FXML
    public void setBoundaries() {
        settings.showSettings();
    }

    @FXML
    public void getAbout() {
        new CreateAboutPanel(true);
    }

    @FXML
    public void hideGraph(ActionEvent e) {
        String id = ((CheckMenuItem) e.getSource()).getId();
        boolean visibility = ((CheckMenuItem) e.getSource()).isSelected();

        switch (id) {
            case "graph1":
                firstGraph.setVisible(!visibility);
                break;
            case "graph2":
                secondGraph.setVisible(!visibility);
                break;
            case "graph3":
                thirdGraph.setVisible(!visibility);
                break;
        }
    }

    @FXML
    public void pauseAplication() {
        ecgData.setAnimation(false);
        VitalLabel.hasOscillate = false;

        defibrilatorHandle.pauseAnimations();
        NIBPhandler.pauseAnimation();

        pauseAplication.setDisable(true);
        resumeAplication.setDisable(false);
    }

    @FXML
    public void resumeAplication() {
        VitalLabel.hasOscillate = true;
        ecgData.setAnimation(true);
        defibrilatorHandle.resumeAnimation();
        NIBPhandler.resumeAnimation();
        pauseAplication.setDisable(false);
        resumeAplication.setDisable(true);
    }

    @FXML
    public void shutDownMonitor() {
        stopMonitor();
        showOnlySideButtons(true);
    }

    private void stopMonitor() {
        setVisibilityAllComponents(false);
        firstGraph.getParent().setStyle("-fx-background-color: black;");
        logoPane.setVisible(true);

        if (ecgData != null) ecgData.clear();
        ecgData = null;

        VitalLabel.vitalData = new VitalData();

        removeGraph(1);
        removeGraph(2);
        removeGraph(3);

        hideDefibrilator();

        ValuesAboutStateOfApplication.clear();
    }

    public void normalMonitorStart(boolean AED) {
        if (AED && powerOn.isVisible()) {
            startPoweringOnAnimation(0.4, true);
            return;
        }

        if (powerOn.isVisible() || defibrilator.isVisible()) {
            return;
        }

        //normalni beh monitoru
        if (NIBPGridPane.isVisible() && !NIBPhandler.isNIBPRunning()) {
            hidePressure();
        }

        defibrilatorHandle.defibStart(AED, manualSetting, AEDfunction);
    }

    /**
     * nastavuje silu vyboje na defibrilatoru
     *
     * @param e tlacitko ktere je zmacknuto
     */
    public void setPowerValue(ActionEvent e) {
        defibrilatorHandle.setActivePowerButton((Button) e.getSource());
    }

    /**
     * reaguje na zmeneni checkboxu s elektrodami
     */
    public void changeElectrods() {
        defibrilatorHandle.changeElectrods();
    }

    /**
     * spusti analyzu pacienta
     */
    @FXML
    public void runAnalyze() {
        if (VitalLabel.vitalData != null && defibrilator.isVisible()) {
            defibrilatorHandle.runAnalyza(VitalLabel.vitalData.getSuccessOfAnalyse());
        }
    }

    /**
     * spousti animaci nacitani energie
     */
    @FXML
    public void reloadDefibrilator() {
        if (!defibrilatorHandle.isAED() && defibrilator.isVisible()) {
            defibrilatorHandle.reloadDefibrilator(true);
        }
    }

    private void setSoundButtonDefibrilator() {
        soundDefibrilator.setOnAction(event -> {
            if (defibrilatorHandle.isActiveSound()) {
                defibrilatorHandle.setActiveSounds(false);
                soundDefibrilator.setStyle("-fx-background-color:white; -fx-text-fill:black;");
                soundDefibrilator.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.GLOBAL_SOUND));
            } else {
                defibrilatorHandle.setActiveSounds(true);
                soundDefibrilator.setStyle("");
                soundDefibrilator.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.DEFIBRILATOR_MUTE));
            }
        });

        if (ValuesAboutStateOfApplication.DEFIBRILATORSound) {
            soundDefibrilator.fire();
        }
    }

    private void setElectrodsCheckboxes() {
        connectTherapyElectrods.setDisable(true);
        connectEcgElectrods.selectedProperty().addListener((observable, oldValue, newValue) -> {
            connectTherapyElectrods.setDisable(!connectEcgElectrods.isSelected());
            connectTherapyElectrods.setSelected(false);
        });
        connectEcgElectrods.setOnAction(event -> {
            defibrilatorHandle.setElectrods(connectEcgElectrods.isSelected(), connectTherapyElectrods.isSelected());
        });
        connectTherapyElectrods.setOnAction(event -> {
            defibrilatorHandle.setElectrods(connectEcgElectrods.isSelected(), connectTherapyElectrods.isSelected());
        });
    }

    /**
     * provede vyboj a zacne pocitat cas od probehnuti vyboje
     */
    @FXML
    public void startShock() {
        defibrilatorHandle.startShock();

        //nastaveni hodnot po uspesne defibrilaci
        if (VitalLabel.vitalData != null && VitalLabel.vitalData.getResuscitationEcg() != -1) {
            ecgData.sendCurve(setEcgData(VitalLabel.vitalData.getResuscitationEcg()));

            VitalLabel.vitalData.setHeartRate(80);
            VitalLabel.vitalData.setPp(35);
            VitalLabel.vitalData.setCo2(58);
            VitalLabel.vitalData.setSpo2(32);
        }
    }

    /**
     * skryje defibrilator z monitoru
     */
    public void hideDefibrilator() {
        ValuesAboutStateOfApplication.isDefibrilatorManualOn = false;
        ValuesAboutStateOfApplication.isDefibrilatorAEDOn = false;
        defibrilatorHandle.hideDefibrilator();
    }

    /*
     ********************************************************************************************************************
     *                                           METODY PRO NIBP
     ********************************************************************************************************************
     */

    /**
     * zobrazi gridpane na mereni tlaku
     */
    private void showPressure() {
        ValuesAboutStateOfApplication.isNIBPON = true;
        NIBPhandler.showPressure();
    }

    /**
     * skryje gridpane na mereni tlaku
     */
    public void hidePressure() {
        ValuesAboutStateOfApplication.isNIBPON = false;
        NIBPhandler.hidePressure();
    }

    /**
     * spousti mereni tlaku
     */
    public void runPressure() {
        if (VitalLabel.vitalData != null) {
            NIBPhandler.runPressure(VitalLabel.vitalData.getSysPressure(), VitalLabel.vitalData.getDiaPressure());
        }
    }

    /**
     * nastavuje mod mereni tlaku reakce na tlacitko mod
     */
    public void setNIBPmode() {
        NIBPhandler.setMode();
    }

    /**
     * zobrazi prehled o jiz probehnutych mereni tlaku
     */
    public void showOverview() {
        if (NIBPOverview.isVisible()) {
            NIBPOverview.setVisible(false);
            overViewButton.setStyle("");
        } else {
            NIBPOverview.setVisible(true);
            NIBPhandler.showOverview();
            overViewButton.setStyle("-fx-background-color:white; -fx-text-fill:black;");
        }
    }

    private void setSoundNIBPButton() {
        soundNIBP.setOnAction(event -> {

            if (NIBPhandler.getVolumeActive()) {
                NIBPhandler.setVolume(0);
                soundNIBP.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.GLOBAL_SOUND));
                soundNIBP.setStyle("-fx-background-color:white; -fx-text-fill:black;");
            } else {
                NIBPhandler.setVolume(5);
                soundNIBP.setText(LoadFXMLWithResourceBundle.getLangResource().getString(KeysForLanguage.NIBP_MUTE));
                soundNIBP.setStyle("");
            }
        });

        if (ValuesAboutStateOfApplication.NIBPSound) {
            soundNIBP.fire();
        }
    }

    /*
     ********************************************************************************************************************
     *                                        METODY PRO ZAPNUTI MONITORU
     ********************************************************************************************************************
     */

    /**
     * spousti animaci na zapinanni monitoru
     *
     * @param time           cas za jak dlouho se ma volat metoda pro naplnovani
     *                       progressbaru
     * @param isDefibrilator true pokud byl zmacknut mod pro defibrilator
     */
    private void startPoweringOnAnimation(double time, boolean isDefibrilator) {
        poweringOnProgress.setVisible(true);
        powerinOnLabel.setVisible(true);

        if (ValuesAboutStateOfApplication.isMonitorTurnON) {
            setVisibilityAllComponents(true);
            return;
        }

        if (poweringOnAnimation == null) {
            poweringOnProgress.setProgress(ValuesAboutStateOfApplication.valuesLoadingMonitor);
            //TODO nastavit cas
            poweringOnAnimation = new Timeline(new KeyFrame(Duration.seconds(time), e -> poweringOnAnimation(isDefibrilator)));
            poweringOnAnimation.setCycleCount(Animation.INDEFINITE);
            poweringOnAnimation.playFromStart();
        }
    }

    /**
     * naplnovani progressbaru na animaci zapinani
     *
     * @param isDefibrilator true pokud byl zmacknut mod pro zapnuti
     *                       defibrilatoru
     */
    private void poweringOnAnimation(boolean isDefibrilator) {
        poweringOnProgress.setProgress(poweringOnProgress.getProgress() + 0.05);
        ValuesAboutStateOfApplication.valuesLoadingMonitor = poweringOnProgress.getProgress();

        if (poweringOnProgress.getProgress() >= 1) {
            poweringOnAnimation.stop();
            poweringOnAnimation = null;
            logoPane.setVisible(false);

            poweringOnProgress.setVisible(false);
            powerinOnLabel.setVisible(false);
            poweringOnProgress.setProgress(0);

            setVisibilityAllComponents(true);
            ValuesAboutStateOfApplication.isMonitorTurnON = true;

            if (isDefibrilator) {
                defibrilatorHandle.defibStart(true, manualSetting, AEDfunction);
            }
        }
    }

    /**
     * zacne zapinat monitor klasickym zpusobem (bez defibrilatoru) reakce na
     * tlacitko powerOnButton
     */
    public void showAppComponents() {
        startPoweringOnAnimation(0.6, false);
    }

    /**
     * nastaveni viditelnosti vsech komponent ktere maji byt zobrazeny na
     * monitoru
     *
     * @param visibility true pokud maji by zobrazeny
     */
    private void setVisibilityAllComponents(boolean visibility) {
        if (visibility) {
            setECGData();
            firstGraph.getParent().setStyle("-fx-background-color:whitesmoke");

        }

        firstGraph.setVisible(visibility);
        secondGraph.setVisible(visibility);
        thirdGraph.setVisible(visibility);
        valuesFromRemoter.setVisible(visibility);
        NIBPGridPane.setVisible(false);
        defibrilator.setVisible(false);
        pauseAplication.setDisable(!visibility);
        resumeAplication.setDisable(true);
        graph1.setDisable(!visibility);
        graph2.setDisable(!visibility);
        powerOn.setVisible(!visibility);
        NIBPButton.setVisible(visibility);
        connectionPane.setVisible(!visibility);
        siideButtonsPane.setVisible(visibility);
    }


    /*
     ********************************************************************************************************************
     *                                        METODY PRO FUNKCE MONITORU
     ********************************************************************************************************************
     */

    /**
     * ulozeni a nastaveni hodnot z obdrzene prepravky
     *
     * @param vd data obdrzena z ovladace
     */
    public void setData(VitalData vd) {
        if (ecgData == null) return;

        if (vd.isChangeOnlyDefibrillator()) {
            defibrilatorHandle.setImpedanceFromRemoter(vd.getImpedance());
            setElectrodes(vd);
            return;
        }

        defibrilatorHandle.setImpedanceFromRemoter(vd.getImpedance());
        setElectrodes(vd);

        int hrBefore = VitalLabel.vitalData.getHeartRate();
        int ecgCurve = VitalLabel.vitalData.getEcg();
        int co2 = VitalLabel.vitalData.getCo2();

        VitalLabel.vitalData.setVitalData(vd);

        if (VitalLabel.vitalData.getHeartRate() != hrBefore)
            ecgData.setHR(vd.getHeartRate());

        setEcgData(ecgCurve);

        if (VitalLabel.vitalData.getCo2() != co2)
            ecgData.getGraphCO2().setCO2(vd.getCo2());
    }

    private void setElectrodes(VitalData vd) {
        if (connectEcgElectrods.isSelected() != vd.getConnectECG() || connectTherapyElectrods.isSelected() != vd.getConnectTerapy()) {
            connectEcgElectrods.setSelected(vd.getConnectECG());
            connectTherapyElectrods.setSelected(vd.getConnectTerapy());
            defibrilatorHandle.setElectrods(vd.getConnectECG(), vd.getConnectTerapy());
        }
    }

    /**
     * Pretranformovani pole do listu.
     *
     * @param indexECGCurve index zvolene krivky
     * @return list dat.
     */
    private ObservableList<ObservableList<ECGPoint>> setEcgData(int indexECGCurve) {
        if (VitalLabel.vitalData.getEcg() == -1) {
            return null;
        }

        VitalLabel.hasOscillate = true;
        ObservableList<ObservableList<ECGPoint>> ecg = FXCollections.observableArrayList();
        Pair<String, List<float[][]>> ecgCurve = loadedECGCurves.get(VitalLabel.vitalData.getEcg());

        if (ecgCurve.getValue() != null) {
            if (VitalLabel.vitalData.getEcg() != indexECGCurve) {
                for (float[][] curves : ecgCurve.getValue()) {
                    ObservableList<ECGPoint> points = FXCollections.observableArrayList();

                    for (float[] item : curves) {
                        points.add(new ECGPoint(item[0], item[1], item[2]));
                    }

                    ecg.add(points);
                }

                ecgData.sendCurve(ecg);
            }
        } else {
            if (ecgCurve.getKey().compareTo("Asystolie") == 0) {
                VitalLabel.hasOscillate = false;
                VitalLabel.vitalData.heartRate.set(0);
                ecgData.setHR(1);
                ecgData.plotAsystolie();
            }
        }

        return ecg;
    }

    /**
     * nastavuje ECG data do jednotlivych grafu
     */
    private void setECGData() {
        if (ecgData == null) {
            ecgData = new ECGData();
            addGraph("ECG1", ecgData.getGraphECG1(), 1);
            // createNormalGraphPane(secondGraph, "ECG2", ecgData.getGraphECG2());
            createNormalGraphPane(secondGraph, "CO2", ecgData.getGraphCO2());
        }
    }

    private void clear() {
        defibrilatorHandle.clear();
        NIBPhandler.clear();
    }

    void closeMonitor() {
        if (connectionThread != null) connectionThread.interrupt();

        if (client != null) {
            client.setMonitClosed(true);
        }

        if (getECGData() != null) {
            getECGData().clear();
            clear();
        }

        VitalLabel.vitalData = new VitalData();
    }
}
